/*  Copyright (C) 2011-2012 National Institute For Space Research (INPE) - Brazil.

    This file is part of the TerraLib - a Framework for building GIS enabled applications.

    TerraLib is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License,
    or (at your option) any later version.

    TerraLib is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with TerraLib. See COPYING. If not, write to
    TerraLib Team at <terralib-team@terralib.org>.
 */

/*!
  \file terralib/qt/widgets/se/GroupingWidget.cpp

  \brief A widget used to build a grouping.
*/

// TerraLib
#include "../../../color/ColorBar.h"
#include "../../../common/Globals.h"
#include "../../../common/STLUtils.h"
#include "../../../dataaccess/dataset/DataSet.h"
#include "../../../dataaccess/dataset/DataSetType.h"
#include "../../../dataaccess/utils/Utils.h"
#include "../../../geometry/GeometryProperty.h"
#include "../../../maptools/Enums.h"
#include "../../../maptools/Grouping.h"
#include "../../../maptools/GroupingAlgorithms.h"
#include "../../../maptools/GroupingItem.h"
#include "../../../maptools/Utils.h"
#include "../../../se/Utils.h"
#include "../colorbar/ColorBar.h"
#include "../se/SymbologyPreview.h"
#include "GroupingWidget.h"
#include "ui_GroupingWidgetForm.h"

// STL
#include <cassert>

// QT
#include <QtGui/QMessageBox>

#define MAX_SLICES 200
#define PRECISION 15
#define NO_TITLE "No Value"


te::qt::widgets::GroupingWidget::GroupingWidget(QWidget* parent, Qt::WindowFlags f)
  : QWidget(parent, f),
    m_ui(new Ui::GroupingWidgetForm)
{
  m_ui->setupUi(this);

  QGridLayout* l = new QGridLayout(m_ui->m_colorBarWidget);
  l->setContentsMargins(0,0,0,0);
  m_colorBar = new  te::qt::widgets::colorbar::ColorBar(m_ui->m_colorBarWidget);
  l->addWidget(m_colorBar);

//connects
  connect(m_colorBar, SIGNAL(colorBarChanged()), this, SLOT(onColorBarChanged()));
  connect(m_ui->m_typeComboBox, SIGNAL(activated(int)), this, SLOT(onTypeComboBoxActivated(int)));
  connect(m_ui->m_attrComboBox, SIGNAL(activated(int)), this, SLOT(onAttrComboBoxActivated(int)));
  connect(m_ui->m_applyPushButton, SIGNAL(clicked()), this, SLOT(onApplyPushButtonClicked()));

  initialize();
}

te::qt::widgets::GroupingWidget::~GroupingWidget()
{
  te::common::FreeContents(m_legend);
  m_legend.clear();
}

void te::qt::widgets::GroupingWidget::setLayer(te::map::AbstractLayerPtr layer)
{
  m_layer = layer;

  //set data type
  setDataSetType();

  //set grouping
  setGrouping();
}

std::auto_ptr<te::map::Grouping> te::qt::widgets::GroupingWidget::getGrouping()
{
  std::string attr = m_ui->m_attrComboBox->currentText().toStdString();
  int attrIdx =  m_ui->m_attrComboBox->currentIndex();
  int attrType = m_ui->m_attrComboBox->itemData(attrIdx).toInt();

  int index = m_ui->m_typeComboBox->currentIndex();
  int type = m_ui->m_typeComboBox->itemData(index).toInt();

  std::auto_ptr<te::map::Grouping> group(new te::map::Grouping(attr, (te::map::GroupingType)type));

  group->setPropertyType(attrType);

  group->setNumSlices(m_ui->m_slicesSpinBox->value());

  group->setPrecision(m_ui->m_precSpinBox->value());

  group->setStdDeviation(m_ui->m_stdDevDoubleSpinBox->value());

  std::vector<te::map::GroupingItem*> groupingItems;
  for(std::size_t t = 0; t < m_legend.size(); ++t)
  {
    te::map::GroupingItem* gi = new te::map::GroupingItem(*m_legend[t]);

    groupingItems.push_back(gi);
  }
  group->setGroupingItems(groupingItems);

  return group;
}

void te::qt::widgets::GroupingWidget::initialize()
{
  // create color bar
  m_cb = new te::color::ColorBar(te::color::RGBAColor(255, 0, 0, TE_OPAQUE), te::color::RGBAColor(0, 0, 0, TE_OPAQUE), 256);

  m_colorBar->setHeight(20);
  m_colorBar->setColorBar(m_cb);
  m_colorBar->setScaleVisible(false);

  // fill grouping type combo box
  m_ui->m_typeComboBox->addItem(tr("Equal Steps"), te::map::EQUAL_STEPS);
  m_ui->m_typeComboBox->addItem(tr("Quantil"), te::map::QUANTIL);
  m_ui->m_typeComboBox->addItem(tr("Standard Deviation"), te::map::STD_DEVIATION);
  m_ui->m_typeComboBox->addItem(tr("Unique Value"), te::map::UNIQUE_VALUE);

  onTypeComboBoxActivated(0);

  //set number of slices
  m_ui->m_slicesSpinBox->setMinimum(1);
  m_ui->m_slicesSpinBox->setMaximum(MAX_SLICES);
  m_ui->m_slicesSpinBox->setValue(5);
  m_ui->m_slicesSpinBox->setSingleStep(1);

  //set standard deviation values
  m_ui->m_stdDevDoubleSpinBox->setMinimum(0.25);
  m_ui->m_stdDevDoubleSpinBox->setMaximum(1.0);
  m_ui->m_stdDevDoubleSpinBox->setValue(0.5);
  m_ui->m_stdDevDoubleSpinBox->setSingleStep(0.25);

  //set number of precision
  m_ui->m_precSpinBox->setMinimum(1);
  m_ui->m_precSpinBox->setMaximum(PRECISION);
  m_ui->m_precSpinBox->setValue(6);
  m_ui->m_precSpinBox->setSingleStep(1);

  //adjust table
  m_ui->m_tableWidget->resizeColumnsToContents();
  m_ui->m_tableWidget->horizontalHeader()->setResizeMode(QHeaderView::Stretch);

  m_manual = false;
}

void te::qt::widgets::GroupingWidget::updateUi()
{
  disconnect(m_ui->m_tableWidget, SIGNAL(itemChanged(QTableWidgetItem*)), this, SLOT(onTableWidgetItemChanged(QTableWidgetItem*)));

  m_ui->m_tableWidget->setRowCount(0);

  int index = m_ui->m_typeComboBox->currentIndex();
  int type = m_ui->m_typeComboBox->itemData(index).toInt();

  if(type == te::map::EQUAL_STEPS || type == te::map::QUANTIL || type == te::map::STD_DEVIATION)
  {
    QStringList list;
    list.append(tr("Symbol"));
    list.append(tr("Title"));
    list.append(tr("Min"));
    list.append(tr("Max"));
    list.append(tr("Count"));

    m_ui->m_tableWidget->setColumnCount(5);
    m_ui->m_tableWidget->setHorizontalHeaderLabels(list);
  }
  else if(type == te::map::UNIQUE_VALUE)
  {
    QStringList list;
    list.append(tr("Symbol"));
    list.append(tr("Title"));
    list.append(tr("Value"));
    list.append(tr("Count"));

    m_ui->m_tableWidget->setColumnCount(4);
    m_ui->m_tableWidget->setHorizontalHeaderLabels(list);
  }

  for(std::size_t t = 0; t < m_legend.size(); ++t)
  {
    te::map::GroupingItem* gi = m_legend[t];

    int newrow = m_ui->m_tableWidget->rowCount();
    m_ui->m_tableWidget->insertRow(newrow);

    //symbol
    {
      const std::vector<te::se::Symbolizer*>& ss = gi->getSymbolizers();
      QPixmap pix = te::qt::widgets::SymbologyPreview::build(ss, QSize(24, 24));
      QIcon icon(pix);
      QTableWidgetItem* item = new QTableWidgetItem(icon, "");
      item->setFlags(Qt::ItemIsEnabled);
      m_ui->m_tableWidget->setItem(newrow, 0, item);
    }

    //title
    {
      QTableWidgetItem* item = new QTableWidgetItem(QString::fromStdString(gi->getTitle()));
      item->setFlags(Qt::ItemIsEnabled | Qt::ItemIsEditable);
      m_ui->m_tableWidget->setItem(newrow, 1, item);
    }


    if(type == te::map::EQUAL_STEPS || type == te::map::QUANTIL || type == te::map::STD_DEVIATION)
    {
      if(gi->getTitle() != NO_TITLE)
      {
        //Min
        {
          QTableWidgetItem* item = new QTableWidgetItem(QString::fromStdString(gi->getLowerLimit()));
          item->setFlags(Qt::ItemIsEnabled | Qt::ItemIsEditable);
          m_ui->m_tableWidget->setItem(newrow, 2, item);
        }

        //Max
        {
          QTableWidgetItem* item = new QTableWidgetItem(QString::fromStdString(gi->getUpperLimit()));
          item->setFlags(Qt::ItemIsEnabled | Qt::ItemIsEditable);
          m_ui->m_tableWidget->setItem(newrow, 3, item);
        }
      }

      //Count
      QTableWidgetItem* item = new QTableWidgetItem(QString::number(gi->getCount()));
      item->setFlags(Qt::ItemIsEnabled);
      m_ui->m_tableWidget->setItem(newrow, 4, item);

    }
    else if(type == te::map::UNIQUE_VALUE)
    {
      if(gi->getTitle() != NO_TITLE)
      {
        //Value
        QTableWidgetItem* item = new QTableWidgetItem(QString::fromStdString(gi->getValue()));
        item->setFlags(Qt::ItemIsEnabled);
        m_ui->m_tableWidget->setItem(newrow, 2, item);
      }

      //Count
      QTableWidgetItem* item = new QTableWidgetItem(QString::number(gi->getCount()));
      item->setFlags(Qt::ItemIsEnabled);
      m_ui->m_tableWidget->setItem(newrow, 3, item);

    }
  }

  m_ui->m_tableWidget->resizeColumnsToContents();
  m_ui->m_tableWidget->horizontalHeader()->setResizeMode(QHeaderView::Stretch);

  connect(m_ui->m_tableWidget, SIGNAL(itemChanged(QTableWidgetItem*)), this, SLOT(onTableWidgetItemChanged(QTableWidgetItem*)));
}

void te::qt::widgets::GroupingWidget::setDataSetType()
{
  listAttributes();
}

void te::qt::widgets::GroupingWidget::setGrouping()
{
  te::map::Grouping* grouping = m_layer->getGrouping();

  if(!grouping)
    return;

  //type
  te::map::GroupingType type = grouping->getType();
  
  for(int i = 0; i < m_ui->m_typeComboBox->count(); ++i)
  {
    if(type == m_ui->m_typeComboBox->itemData(i).toInt())
    {
      m_ui->m_typeComboBox->setCurrentIndex(i);
      break;
    }
  }

  //attr name
  std::string attrName = grouping->getPropertyName();

  for(int i = 0; i < m_ui->m_attrComboBox->count(); ++i)
  {
    if(attrName == m_ui->m_attrComboBox->itemText(i).toStdString())
    {
      m_ui->m_attrComboBox->setCurrentIndex(i);
      break;
    }
  }

  //precision
  size_t prec = grouping->getPrecision();

  m_ui->m_precSpinBox->setValue((int)prec);

  //slices
  size_t slices = grouping->getNumSlices();

  m_ui->m_slicesSpinBox->setValue((int)slices);

  //std dev
  float stdDev = grouping->getStdDeviation();

  m_ui->m_stdDevDoubleSpinBox->setValue((double)stdDev);

  //grouping items
  te::common::FreeContents(m_legend);
  m_legend.clear();

  for(size_t t = 0; t < grouping->getGroupingItems().size(); ++t)
  {
    te::map::GroupingItem* gi = new te::map::GroupingItem(*grouping->getGroupingItems()[t]);

    m_legend.push_back(gi);
  }

  updateUi();
}

void te::qt::widgets::GroupingWidget::onApplyPushButtonClicked()
{
  if(m_manual)
  {
      int reply = QMessageBox::question(this, tr("Classification"), tr("Manual changes will be lost. Continue?"), QMessageBox::Yes | QMessageBox::Cancel);

      if(reply != QMessageBox::Yes)
        return;
  }

  int index = m_ui->m_typeComboBox->currentIndex();

  int type = m_ui->m_typeComboBox->itemData(index).toInt();
  int slices = m_ui->m_slicesSpinBox->value();
  int prec = m_ui->m_precSpinBox->value();
  double stdDev = m_ui->m_stdDevDoubleSpinBox->value();

  std::string attr = m_ui->m_attrComboBox->currentText().toStdString();
  int attrIdx =  m_ui->m_attrComboBox->currentIndex();
  int attrType = m_ui->m_attrComboBox->itemData(attrIdx).toInt();

  te::common::FreeContents(m_legend);
  m_legend.clear();

  std::string mean = "";

  int nullValues = 0;

  if(type == te::map::EQUAL_STEPS)
  {
    std::vector<double> vec;

    getDataAsDouble(vec, attr, attrType, nullValues);

    te::map::GroupingByEqualSteps(vec.begin(), vec.end(), slices, m_legend, prec);

    buildSymbolizer(mean);

    createDoubleNullGroupingItem(nullValues);
  }
  else if(type == te::map::QUANTIL) 
  {
    std::vector<double> vec;

    getDataAsDouble(vec, attr, attrType, nullValues);

    te::map::GroupingByQuantil(vec.begin(), vec.end(), slices, m_legend, prec);

    buildSymbolizer(mean);

    createDoubleNullGroupingItem(nullValues);
  }
  else if(type == te::map::STD_DEVIATION) 
  {
    std::vector<double> vec;

    getDataAsDouble(vec, attr, attrType, nullValues);

    te::map::GroupingByStdDeviation(vec.begin(), vec.end(), stdDev, m_legend, mean, prec);

    buildSymbolizer(mean);

    createDoubleNullGroupingItem(nullValues);
  }
  else if(type == te::map::UNIQUE_VALUE) 
  {
    std::vector<std::string> vec;

    getDataAsString(vec, attr, nullValues);

    te::map::GroupingByUniqueValues(vec, attrType, m_legend, prec);

    buildSymbolizer(mean);

    createStringNullGroupingItem(nullValues);
  }

  updateUi();

  m_manual = false;

  emit applyPushButtonClicked();
}

void te::qt::widgets::GroupingWidget::onTypeComboBoxActivated(int idx)
{
  int type = m_ui->m_typeComboBox->itemData(idx).toInt();

  if(type == te::map::EQUAL_STEPS)
  {
    m_ui->m_slicesSpinBox->setEnabled(true);
    m_ui->m_precSpinBox->setEnabled(true);
    m_ui->m_stdDevDoubleSpinBox->setEnabled(false);
  }
  else if(type == te::map::QUANTIL) 
  {
    m_ui->m_slicesSpinBox->setEnabled(true);
    m_ui->m_precSpinBox->setEnabled(true);
    m_ui->m_stdDevDoubleSpinBox->setEnabled(false);
  }
  else if(type == te::map::STD_DEVIATION) 
  {
    m_ui->m_slicesSpinBox->setEnabled(false);
    m_ui->m_precSpinBox->setEnabled(true);
    m_ui->m_stdDevDoubleSpinBox->setEnabled(true);
  }
  else if(type == te::map::UNIQUE_VALUE) 
  {
    m_ui->m_slicesSpinBox->setEnabled(false);
    m_ui->m_precSpinBox->setEnabled(true);
    m_ui->m_stdDevDoubleSpinBox->setEnabled(false);
  }

  if(m_layer.get())
    listAttributes();
}

void te::qt::widgets::GroupingWidget::onAttrComboBoxActivated(int idx)
{
//  int attrType = m_ui->m_attrComboBox->itemData(idx).toInt();
}

void te::qt::widgets::GroupingWidget::onColorBarChanged()
{
  if(m_layer.get())
  {
    buildSymbolizer();

    updateUi();
  }
}

void  te::qt::widgets::GroupingWidget::onTableWidgetItemChanged(QTableWidgetItem* item)
{
  int index = m_ui->m_typeComboBox->currentIndex();
  int type = m_ui->m_typeComboBox->itemData(index).toInt();

  int curRow = m_ui->m_tableWidget->currentRow();
  int curCol = m_ui->m_tableWidget->currentColumn();

  QString str = item->text();

  if(curCol == 1) // title
  {
    m_legend[curRow]->setTitle(str.toStdString());

    m_manual = true;
  }
  else if(curCol == 2 || curCol == 3) // min and max
  {
    if(type == te::map::EQUAL_STEPS || type == te::map::QUANTIL || type == te::map::STD_DEVIATION)
    {
      bool ok = false;

      str.toDouble(&ok);

      if(!ok)
      {
        if(curCol == 2)
          item->setText(m_legend[curRow]->getLowerLimit().c_str());
        else if(curCol ==3)
          item->setText(m_legend[curRow]->getUpperLimit().c_str());
      }
      else
      {
        if(curCol == 2)
          m_legend[curRow]->setLowerLimit(item->text().toStdString());
        else if(curCol ==3)
          m_legend[curRow]->setUpperLimit(item->text().toStdString());

        m_manual = true;
      }
    }
  }
}

void te::qt::widgets::GroupingWidget::getDataAsDouble(std::vector<double>& vec, const std::string& attrName, const int& dataType, int& nullValues)
{
  assert(m_layer.get());

  std::auto_ptr<te::map::LayerSchema> dsType(m_layer->getSchema());

  std::size_t idx;

  for(std::size_t t = 0; t < dsType->getProperties().size(); ++t)
  {
    if(dsType->getProperty(t)->getName() == attrName)
    {
      idx = t;
      break;
    }
  }

  std::auto_ptr<te::da::DataSet> ds(m_layer->getData());

  ds->moveBeforeFirst();

  while(ds->moveNext())
  {
    if(ds->isNull(idx))
    {
      ++nullValues;
      continue;
    }

    if(dataType == te::dt::INT16_TYPE)
      vec.push_back((double)ds->getInt16(idx));
    else if(dataType == te::dt::INT32_TYPE)
      vec.push_back((double)ds->getInt32(idx));
    else if(dataType == te::dt::INT64_TYPE)
      vec.push_back((double)ds->getInt64(idx));
    else if(dataType == te::dt::FLOAT_TYPE)
      vec.push_back((double)ds->getFloat(idx));
    else if(dataType == te::dt::DOUBLE_TYPE)
      vec.push_back(ds->getDouble(idx));
    else if(dataType == te::dt::NUMERIC_TYPE)
    {
      QString strNum = ds->getNumeric(idx).c_str();

      bool ok = false;

      double value = strNum.toDouble(&ok);

      if(ok)
        vec.push_back(value);
    }
  }
}

void te::qt::widgets::GroupingWidget::getDataAsString(std::vector<std::string>& vec, const std::string& attrName, int& nullValues)
{
  assert(m_layer.get());

  std::auto_ptr<te::map::LayerSchema> dsType(m_layer->getSchema());

  std::size_t idx;

  for(std::size_t t = 0; t < dsType->getProperties().size(); ++t)
  {
    if(dsType->getProperty(t)->getName() == attrName)
    {
      idx = t;
      break;
    }
  }

  std::auto_ptr<te::da::DataSet> ds(m_layer->getData());

  ds->moveBeforeFirst();

  while(ds->moveNext())
  {
    if(!ds->isNull(idx))
      vec.push_back(ds->getAsString(idx));
    else
      ++nullValues;
  }
}

void te::qt::widgets::GroupingWidget::createDoubleNullGroupingItem(int count)
{
  if(count == 0)
    return;

  te::map::GroupingItem* legendItem = new te::map::GroupingItem;
  legendItem->setLowerLimit(te::common::Globals::sm_nanStr);
  legendItem->setUpperLimit(te::common::Globals::sm_nanStr);
  legendItem->setTitle(NO_TITLE);
  legendItem->setCount(count);

  int geomType = getGeometryType();
  std::vector<te::se::Symbolizer*> symbVec;
  te::se::Symbolizer* s = te::se::CreateSymbolizer((te::gm::GeomType)geomType, "#dddddd");
  symbVec.push_back(s);
  legendItem->setSymbolizers(symbVec);

  m_legend.push_back(legendItem);
}

void te::qt::widgets::GroupingWidget::createStringNullGroupingItem(int count)
{
  if(count == 0)
    return;

  te::map::GroupingItem* legendItem = new te::map::GroupingItem;
  legendItem->setValue(te::common::Globals::sm_nanStr);
  legendItem->setTitle(NO_TITLE);
  legendItem->setCount(count);

  int geomType = getGeometryType();
  std::vector<te::se::Symbolizer*> symbVec;
  te::se::Symbolizer* s = te::se::CreateSymbolizer((te::gm::GeomType)geomType, "#dddddd");
  symbVec.push_back(s);
  legendItem->setSymbolizers(symbVec);

  m_legend.push_back(legendItem);
}

int te::qt::widgets::GroupingWidget::getGeometryType()
{
  assert(m_layer.get());

  return te::map::GetGeomType(m_layer);
}

void te::qt::widgets::GroupingWidget::buildSymbolizer(std::string meanTitle)
{
  int legendSize = m_legend.size();

  std::vector<te::color::RGBAColor> colorVec;
  
  if(meanTitle.empty())
  {
    colorVec = m_cb->getSlices(legendSize);
  }
  else
  {
    int beforeMean = 0;
    int afterMean = 0;
    int meanIdx = 0;

    for(size_t t = 0; t < m_legend.size(); ++t)
    {
      if(m_legend[t]->getTitle() != meanTitle)
      {
        beforeMean++;
      }
      else
      {
        meanIdx = t;
        afterMean = m_legend.size() - t - 1;
        break;
      }
    }

    std::vector<te::color::RGBAColor> lowerColorVec = m_cb->getLowerMeanSlices(beforeMean);
    te::color::RGBAColor meanColor = m_cb->getMeanSlice();
    std::vector<te::color::RGBAColor> upperColorVec = m_cb->getUpperMeanSlices(afterMean);

    for(size_t t = 0; t < lowerColorVec.size(); ++t)
      colorVec.push_back(lowerColorVec[t]);

    colorVec.push_back(meanColor);

    for(size_t t = 0; t < upperColorVec.size(); ++t)
      colorVec.push_back(upperColorVec[t]);
  }

  if(colorVec.size() != m_legend.size())
    return;

  int geomType = getGeometryType();

  for(size_t t = 0; t < colorVec.size(); ++t)
  {
    std::vector<te::se::Symbolizer*> symbVec;

    te::se::Symbolizer* s = te::se::CreateSymbolizer((te::gm::GeomType)geomType, colorVec[t].getColor());

    symbVec.push_back(s);

    m_legend[t]->setSymbolizers(symbVec);
  }
}

void te::qt::widgets::GroupingWidget::listAttributes()
{
  QString curValue = m_ui->m_attrComboBox->currentText();

  m_ui->m_attrComboBox->clear();

  std::auto_ptr<te::map::LayerSchema> dsType(m_layer->getSchema());

  //grouping type
  int index = m_ui->m_typeComboBox->currentIndex();
  int type = m_ui->m_typeComboBox->itemData(index).toInt();

  if(type == te::map::EQUAL_STEPS ||type == te::map::QUANTIL || type == te::map::STD_DEVIATION)
  {
    for(size_t t = 0; t < dsType->getProperties().size(); ++t)
    {
      te::dt::Property* p = dsType->getProperty(t);

      int dataType = p->getType();

      switch(dataType)
      {
        case te::dt::INT16_TYPE:
        case te::dt::INT32_TYPE:
        case te::dt::INT64_TYPE:
        case te::dt::FLOAT_TYPE:
        case te::dt::DOUBLE_TYPE:
        case te::dt::NUMERIC_TYPE:
          m_ui->m_attrComboBox->addItem(p->getName().c_str(), p->getType());

        default:
          continue;
      }
    }
  }
  else if(type == te::map::UNIQUE_VALUE) 
  {
    for(size_t t = 0; t < dsType->getProperties().size(); ++t)
    {
      te::dt::Property* p = dsType->getProperty(t);

      int dataType = p->getType();

      switch(dataType)
      {
        case te::dt::INT16_TYPE:
        case te::dt::INT32_TYPE:
        case te::dt::INT64_TYPE:
        case te::dt::FLOAT_TYPE:
        case te::dt::DOUBLE_TYPE:
        case te::dt::NUMERIC_TYPE:
        case te::dt::STRING_TYPE:
          m_ui->m_attrComboBox->addItem(p->getName().c_str(), p->getType());

        default:
          continue;
      }
    }
  }

  if(curValue.isEmpty() == false)
  {
    int idx = m_ui->m_attrComboBox->findText(curValue);

    if(idx != -1)
      m_ui->m_attrComboBox->setCurrentIndex(idx);
  }
}
